// this file is used to handle the outpost tasks for transporting
// wood, gold, and food back to the main maps

// HANDLES OUTPOST TASK AND BUTTON INITIALIZATION

// when the tasks start, ensure that all production using the
// oupost is disabled (will be enabled after the tasks are complete)
function OutpostTaskInit::onExit(%vertex)
{
   // disable all of the outpost production
   tsEnableCommand(ProduceGoldCmdData, false);
   tsEnableCommand(ProduceWoodCmdData, false);
   tsEnableCommand(ProduceFoodCmdData, false);
}

// HANDLES OUTPOST TASK ENABLING AND DISABLING

// every time an outpost is created on the player's team, ensure
// that the outpost tasks are enabled
function OutpostTaskCreate::onMessage(%vertex, %type, %param)
{
   // check if an outpost is created, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }
   
   // pretent that this vertex was not affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskCreate", $TSV_AVAIL);
   
   // clear existing task creation lists
   %nameList = "";
   %messageList = "";
   
   // check if the gold task needs to be added
   if (OutpostTaskGoldTrigger.isUnavailable() == true)
   {
      %nameList = %nameList @ "ID_TASKNAME_OUTPOSTGOLD ";
      %messageList = %messageList @ "id_outpost_gold ";
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskGoldTrigger", $TSV_AVAIL);
   }
   
   // check if the wood task needs to be added
   if (OutpostTaskWoodTrigger.isUnavailable() == true)
   {
      %nameList = %nameList @ "ID_TASKNAME_OUTPOSTWOOD ";
      %messageList = %messageList @ "id_outpost_wood ";
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskWoodTrigger", $TSV_AVAIL);
   }
   
   // check if the food task needs to be added
   if (OutpostTaskFoodTrigger.isUnavailable() == true)
   {
      %nameList = %nameList @ "ID_TASKNAME_OUTPOSTFOOD ";
      %messageList = %messageList @ "id_outpost_food";
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskFoodTrigger", $TSV_AVAIL);
   }
   
   // if no tasks need to be updated, they are already completed
   // or active-- done
   %count = getWordCount(%messageList);
   if (%count == 0)
   {
      return;
   }
   
   // create the character that the alert will be attached to
   %posX = getWord(mapPlatform.position, 0);
   %posY = getWord(mapPlatform.position, 1);
   //tsSkinObj("outpostCiv", "banker");
   if(isObject("outpostCiv")==false)
   {
      %outpostCiv = tsCreateObj("MaleCiv", "outpostCiv", "0", %posX @ " " @ %posY, "outpostCiv");
      %outpostCiv.showMesh(false);
      %outpostCiv.name = "Outpost Worker";
   }

   // make all of the outpost satellites visible
   mapsatellite_outpostfood.visible = true;
   mapsatellite_outpostwood.visible = true;
   mapsatellite_outpostgold.visible = true;
   
   // for each of the messages that need to be sent, set up the message
   // and send the alert to the character that was created
   for (%index = 0; %index < %count; %index++)
   {
      %name = getWord(%nameList, %index);
      %message = getWord(%messageList, %index);
      SendOutpostTaskAlert(%name, %message);
   }
}

// when an alert message needs to be sent to the outpost civilian
function SendOutpostTaskAlert(%name, %message)
{
   %dialog = tsCreateTaskDlgData(%name, true, $TaskButton::Accept, %message);
   tsAddTaskDialog(%message, %dialog);
   tsAddAcceptCost($Resource::Gold, 200, %dialog);
   tsAddAcceptCost($Resource::Wood, 200, %dialog);
   tsAddAcceptCost($Resource::Food, 20, %dialog);
   tsSendTaskDlgMsg(%dialog, "outpostCiv");
   tsDestroyTaskDlgData(%dialog);
}

// every time a platform is created, this happens when the platform
// transitions teams or was destroyed/downgraded
function OutpostTaskDestroy::onMessage(%vertex, %type, %param)
{
   // check if a platform is created, and if not, do nothing
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }

   // pretend that this vertex was not affected (needs to be immediately
   // resolved, so the vertex is always active to receive messages)
   %vertex.decCompleteCount();
   TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskDestroy", $TSV_AVAIL);
   
   // if there is no object to destroy, we are done
   if (isObject(outpostCiv) == false)
   {
      return;
   }
   
   // remove all of the tasks from the outpost character, then delete
   outpostCiv.performAction("die");
   
   // check if the gold task needs to be disabled
   if (OutpostTaskGoldTrigger.isAvailable() == true)
   {
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskGoldTrigger", $TSV_UNAVAIL);
   }
   
   // check if the wood task needs to be disabled
   if (OutpostTaskWoodTrigger.isAvailable() == true)
   {
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskWoodTrigger", $TSV_UNAVAIL);
   }
   
   // check if the food task needs to be disabled
   if (OutpostTaskFoodTrigger.isAvailable() == true)
   {
      TaskModifier.strMarkVertex("OutpostTasks", "OutpostTaskFoodTrigger", $TSV_UNAVAIL);
   }
}

// HANDLES OUTPOST GOLD, WOOD, AND FOOD MISSION STARTS

// when the gold outpost mission has been selected
function OutpostTaskGoldTrigger::onExit(%vertex)
{
   // Create outpost gold scenario
   %scenario = new SLScenario()
   {
      stack   = true;
      mapMain = false;
      mapName = "assets/data/missions/sandlot/resourcegold.mis";
      mapMinimap = "assets/client/ui/minimaps/resourcegoldminimap.png";
      
      // base resources for starting the tutorial
      resourceGold = "250 0";
      resourceWood = "250 0";
      resourceFood = "20 0";
      resourceFoodMax = "20";
      resourceWater = "0 0";
      resourceWaterMax = "0";
      resourceHappiness = "80 0";
      
      // turn off all disasters
      disasterStack = $DisasterStack::Push;
      addDisasterNode = "0 60 0";
   };
   
   // update the disasters that will be used
   UpdateOutpostDisasters(%scenario);
   
   // enable gold production
   tsEnableCommand(ProduceGoldCmdData, true);
   
   // if none of the other building commands are enabled, select the gold
   if (csIsCommandDisabled(ProduceWoodCmdData) == true &&
      csIsCommandDisabled(ProduceFoodCmdData) == true)
   {
      csExecuteProductionSwapCmdBtn(gold, ProduceGoldCmdData);
   }
   
   // load the scenario
   LoadScenario(%scenario, "", startOutpostGoldTasks, quitOutpostGoldTasks);
}

// when the wood outpost mission has been selected
function OutpostTaskWoodTrigger::onExit(%vertex)
{
   // Create outpost wood scenario
   %scenario = new SLScenario()
   {
      stack   = true;
      mapMain = false;
      mapName = "assets/data/missions/sandlot/resourcewood.mis";
      mapMinimap = "assets/client/ui/minimaps/resourcewoodminimap.png";
      
      // base resources for starting the tutorial
      resourceGold = "250 0";
      resourceWood = "250 0";
      resourceFood = "20 0";
      resourceFoodMax = "20";
      resourceWater = "0 0";
      resourceWaterMax = "0";
      resourceHappiness = "80 0";
      
      // turn off all disasters
      disasterStack = $DisasterStack::Push;
      addDisasterNode = "0 60 0";
   };
   
   // update the disasters that will be used
   UpdateOutpostDisasters(%scenario);
   
   // enable wood production
   tsEnableCommand(ProduceWoodCmdData, true);
   
   // if none of the other building commands are enabled, select the wood
   if (csIsCommandDisabled(ProduceGoldCmdData) == true &&
      csIsCommandDisabled(ProduceFoodCmdData) == true)
   {
      csExecuteProductionSwapCmdBtn(wood, ProduceWoodCmdData);
   }
   
   // load the scenario
   LoadScenario(%scenario, "", startOutpostWoodTasks, quitOutpostWoodTasks);
}

// when the food outpost mission has been selected
function OutpostTaskFoodTrigger::onExit(%vertex)
{
   // Create outpost food scenario
   %scenario = new SLScenario()
   {
      stack   = true;
      mapMain = false;
      mapName = "assets/data/missions/sandlot/resourcefood.mis";
      mapMinimap = "assets/client/ui/minimaps/resourcefoodminimap.png";
      
      // base resources for starting the tutorial
      resourceGold = "250 0";
      resourceWood = "250 0";
      resourceFood = "20 0";
      resourceFoodMax = "20";
      resourceWater = "0 0";
      resourceWaterMax = "0";
      resourceHappiness = "80 0";
      
      // turn off all disasters
      disasterStack = $DisasterStack::Push;
      addDisasterNode = "0 60 0";
   };
   
   // update the disasters that will be used
   UpdateOutpostDisasters(%scenario);
   
   // enable food production
   tsEnableCommand(ProduceFoodCmdData, true);
   
   // if none of the other building commands are enabled, select the food
   if (csIsCommandDisabled(ProduceGoldCmdData) == true &&
      csIsCommandDisabled(ProduceWoodCmdData) == true)
   {
      csExecuteProductionSwapCmdBtn(wood, ProduceFoodCmdData);
   }
   
   // load the scenario
   LoadScenario(%scenario, "", startOutpostFoodTasks, quitOutpostFoodTasks);
}

// HANDLES OUTPOST MISSION DISASTER SETTINGS

// each outpost scenario should get progressively harder, so determine
// the level of difficulty for the disasters that should occur on the
// outpost map and update the scenario
function UpdateOutpostDisasters(%scenario)
{
   // determine the difficulty of the next disaster based on the
   // number of outpost missions that have already been completed
   %difficulty = 0;
   if (OutpostTaskGoldTrigger.isAvailable() == true)
   {
      %difficulty++;
   }
   if (OutpostTaskWoodTrigger.isAvailable() == true)
   {
      %difficulty++;
   }
   if (OutpostTaskFoodTrigger.isAvailable() == true)
   {
      %difficulty++;
   }
   
   switch(%difficulty)
   {
   // lowest difficulty
   case 2:
      UpdateOutpostDisastersEasy(%scenario);
   
   // medium difficulty
   case 1:
      UpdateOutpostDisastersMedium(%scenario);
   
   // hardest difficulty
   case 0:
      UpdateOutpostDisastersHard(%scenario);
   }
}

// when an outpost mission's disaster level should be set to easy
function UpdateOutpostDisastersEasy(%scenario)
{
   // first create the disasters for this difficult level
   %scenario.addDisasterNode = "0 120 1";
   
   // plagues
   %scenario.disasterPlagueScenario = new SLPlagueScenario()
   {
      chance = 15;
      addRequirement = Farm;
      addRequirement = Ranch;
      duration = 30;
      
      produceRate = 0.5;
      addProduce = "wheat";
      addProduce = "corn";
      addProduce = "pumpkin";
      addProduce = "chicken";
      addProduce = "pig";
      addProduce = "cow";
   };
   
   // famines
   %scenario.disasterFamineScenario = new SLFamineScenario()
   {
      chance = 10;
      duration = 0;
      
      resource = $Resource::Food;
      cost = 0.50;
   };
   
   // droughts
   %scenario.disasterDroughtScenario = new SLDroughtScenario()
   {
      chance = 15;
      addRequirement = well;
      duration = 45;
      
      cost = 0.90;
   };
   
   // bandit attacks
   %scenario.disasterBanditScenario = new SLBanditAttackScenario()
   {
      chance = 15;
      duration = 1000;
      
      fadeTime = 1;
      goldMin = 0;
      goldMax = 100;
      goldLow = 5;
      goldHigh = 10;
      lowCount = 1;
      highCount = 2;
      
      happinessCount = -25;
      happinessIncrease = 0;
      happinessPlateau = 10;
      happinessDecrease = 0;
   };
   
   // tornados
   %scenario.disasterTornadoScenario = new SLTornadoScenario()
   {
      object = "tornado";

      chance = 15;
      duration = 60;

      travelAngle = 35;
      travelRandom = 25;
      travelTimeLow = 3;
      travelTimeHigh = 6;
      travelSpeedLow = 3;
      travelSpeedHigh = 5;
      radius = 8;
      damage = 25;
   };
   
   // fires
   %scenario.disasterFireScenario = new SLFireScenario()
   {
      chance = 15;
      duration = -1;
      
      startStrength = 2;
      
      spreadCount = 2;
      spreadTime = 15;
      spreadChance = 30;
      spreadRadius = 25;
      spreadStrength = 1;
            
      damageTime = 0.5;
      damageStrength = 2;
   };
   
   // earthquakes
   %scenario.disasterEarthquakeScenario = new SLEarthquakeScenario()
   {
      chance = 15;
      duration = 5;
      
      shakeDistance = 0.4;
      damageLow = 20;
      damageHigh = 35;
   };
   
   // set up all of the disaster rates for this difficulty level
   %scenario.disasterDroughtRate = "15 100";
   %scenario.disasterPlagueRate = "10 60";
   %scenario.disasterFamineRate = "15 100";
   %scenario.disasterFireRate = "15 120";
   %scenario.disasterBanditRate = "15 120";
   %scenario.disasterEarthquakeRate = "15 160";
   %scenario.disasterTornadoRate = "15 120";
}

// when an outpost mission's disaster level should be set to medium
function UpdateOutpostDisastersMedium(%scenario)
{
   // first create the disasters for this difficult level
   %scenario.addDisasterNode = "0 100 1";
   //disasterStack = $DisasterStack::Push;
   
   // plagues
   %scenario.disasterPlagueScenario = new SLPlagueScenario()
   {
      chance = 20;
      addRequirement = Farm;
      addRequirement = Ranch;
      duration = 45;
      
      produceRate = 0.5;
      addProduce = "wheat";
      addProduce = "corn";
      addProduce = "pumpkin";
      addProduce = "chicken";
      addProduce = "pig";
      addProduce = "cow";
   };
   
   // famines
   %scenario.disasterFamineScenario = new SLFamineScenario()
   {
      chance = 15;
      duration = 0;
      
      resource = $Resource::Food;
      cost = 0.50;
   };
   
   // droughts
   %scenario.disasterDroughtScenario = new SLDroughtScenario()
   {
      chance = 20;
      addRequirement = well;
      duration = 45;
      
      cost = 0.90;
   };
   
   // bandit attacks
   %scenario.disasterBanditScenario = new SLBanditAttackScenario()
   {
      chance = 20;
      duration = 1000;
      
      fadeTime = 1;
      goldMin = 0;
      goldMax = 100;
      goldLow = 5;
      goldHigh = 10;
      lowCount = 1;
      highCount = 3;
      
      happinessCount = -25;
      happinessIncrease = 0;
      happinessPlateau = 10;
      happinessDecrease = 0;
   };
   
   // tornados
   %scenario.disasterTornadoScenario = new SLTornadoScenario()
   {
      object = "tornado";

      chance = 15;
      duration = 60;

      travelAngle = 35;
      travelRandom = 25;
      travelTimeLow = 3;
      travelTimeHigh = 6;
      travelSpeedLow = 3;
      travelSpeedHigh = 5;
      radius = 8;
      damage = 25;
   };
   
   // fires
   %scenario.disasterFireScenario = new SLFireScenario()
   {
      chance = 20;
      duration = -1;
      
      startStrength = 2;
      
      spreadCount = 2;
      spreadTime = 15;
      spreadChance = 30;
      spreadRadius = 25;
      spreadStrength = 1;
            
      damageTime = 0.5;
      damageStrength = 2;
   };
   
   // earthquakes
   %scenario.disasterEarthquakeScenario = new SLEarthquakeScenario()
   {
      chance = 10;
      duration = 5;
      
      shakeDistance = 0.4;
      damageLow = 25;
      damageHigh = 35;
   };
   
   // set up all of the disaster rates for this difficulty level
   %scenario.disasterDroughtRate = "10 100";
   %scenario.disasterPlagueRate = "10 60";
   %scenario.disasterFamineRate = "10 100";
   %scenario.disasterFireRate = "20 120";
   %scenario.disasterBanditRate = "20 100";
   %scenario.disasterEarthquakeRate = "15 160";
   %scenario.disasterTornadoRate = "15 100";
}

// when an outpost mission's disaster level should be set to hard
function UpdateOutpostDisastersHard(%scenario)
{
   // first create the disasters for this difficult level
   %scenario.addDisasterNode = "0 80 2";
   
   // plagues
   %scenario.disasterPlagueScenario = new SLPlagueScenario()
   {
      chance = 30;
      addRequirement = Farm;
      addRequirement = Ranch;
      duration = 60;
      
      produceRate = 0.5;
      addProduce = "wheat";
      addProduce = "corn";
      addProduce = "pumpkin";
      addProduce = "chicken";
      addProduce = "pig";
      addProduce = "cow";
      
   };
   
   // famines
   %scenario.disasterFamineScenario = new SLFamineScenario()
   {
      chance = 20;
      duration = 0;
      
      resource = $Resource::Food;
      cost = 0.50;
   };
   
   // droughts
   %scenario.disasterDroughtScenario = new SLDroughtScenario()
   {
      chance = 15;
      addRequirement = well;
      duration = 90;
      
      cost = 0.9;
   };
   
   // bandit attacks
   %scenario.disasterBanditScenario = new SLBanditAttackScenario()
   {
      chance = 30;
      duration = 1000;
      
      fadeTime = 1;
      goldMin = 0;
      goldMax = 100;
      goldLow = 5;
      goldHigh = 10;
      lowCount = 1;
      highCount = 3;
      
      happinessCount = -25;
      happinessIncrease = 0;
      happinessPlateau = 10;
      happinessDecrease = 0;
   };
   
   // tornados
   %scenario.disasterTornadoScenario = new SLTornadoScenario()
   {
      object = "tornado";

      chance = 30;
      duration = 80;

      travelAngle = 35;
      travelRandom = 25;
      travelTimeLow = 3;
      travelTimeHigh = 6;
      travelSpeedLow = 3;
      travelSpeedHigh = 5;
      radius = 8;
      damage = 25;
   };
   
   // fires
   %scenario.disasterFireScenario = new SLFireScenario()
   {
      chance = 35;
      duration = -1;
      
      startStrength = 2;
      
      spreadCount = 2;
      spreadTime = 15;
      spreadChance = 30;
      spreadRadius = 25;
      spreadStrength = 1;
            
      damageTime = 0.5;
      damageStrength = 2;
   };
   
   // earthquakes
   %scenario.disasterEarthquakeScenario = new SLEarthquakeScenario()
   {
      chance = 25;
      duration = 5;
      
      shakeDistance = 0.4;
      damageLow = 25;
      damageHigh = 35;
   };
   
   // set up all of the disaster rates for this difficulty level
   %scenario.disasterDroughtRate = "15 100";
   %scenario.disasterPlagueRate = "10 60";
   %scenario.disasterFamineRate = "15 100";
   %scenario.disasterFireRate = "15 120";
   %scenario.disasterBanditRate = "15 100";
   %scenario.disasterEarthquakeRate = "15 120";
   %scenario.disasterTornadoRate = "15 100";
}
